/*!
 * VERSION: 0.2.1
 * DATE: 2018-02-15
 * UPDATES AND DOCS AT: http://greensock.com
 *
 * @license Copyright (c) 2008-2018, GreenSock. All rights reserved.
 * PixiPlugin is subject to the terms at http://greensock.com/standard-license or for
 * Club GreenSock members, the software agreement that was issued with your membership.
 *
 * @author: Jack Doyle, jack@greensock.com
 */
var _gsScope = (typeof module !== "undefined" && module.exports && typeof global !== "undefined") ? global : this || window;
(_gsScope._gsQueue || (_gsScope._gsQueue = [])).push(function () {
    "use strict";

    var _numExp = /(\d|\.)+/g,
		_relNumExp = /(?:\d|\-\d|\.\d|\-\.\d|\+=\d|\-=\d|\+=.\d|\-=\.\d)+/g,
		_colorLookup = {aqua:[0,255,255],
			lime:[0,255,0],
			silver:[192,192,192],
			black:[0,0,0],
			maroon:[128,0,0],
			teal:[0,128,128],
			blue:[0,0,255],
			navy:[0,0,128],
			white:[255,255,255],
			fuchsia:[255,0,255],
			olive:[128,128,0],
			yellow:[255,255,0],
			orange:[255,165,0],
			gray:[128,128,128],
			purple:[128,0,128],
			green:[0,128,0],
			red:[255,0,0],
			pink:[255,192,203],
			cyan:[0,255,255],
			transparent:[255,255,255,0]},
		_hue = function(h, m1, m2) {
			h = (h < 0) ? h + 1 : (h > 1) ? h - 1 : h;
			return ((((h * 6 < 1) ? m1 + (m2 - m1) * h * 6 : (h < 0.5) ? m2 : (h * 3 < 2) ? m1 + (m2 - m1) * (2 / 3 - h) * 6 : m1) * 255) + 0.5) | 0;
		},
		/**
		 * @private Parses a color (like #9F0, #FF9900, rgb(255,51,153) or hsl(108, 50%, 10%)) into an array with 3 elements for red, green, and blue or if "format" parameter is "hsl", it will populate the array with hue, saturation, and lightness values. Or if "format" is "number", it'll return a number like 0xFF0000 instead of an array. If a relative value is found in an hsl() or hsla() string, it will preserve those relative prefixes and all the values in the array will be strings instead of numbers (in all other cases it will be populated with numbers).
		 * @param {(string|number)} v The value the should be parsed which could be a string like #9F0 or rgb(255,102,51) or rgba(255,0,0,0.5) or it could be a number like 0xFF00CC or even a named color like red, blue, purple, etc.
		 * @param {(string)} format If "hsl", an hsl() or hsla() value will be returned instead of rgb() or rgba(). Or if "number", then a numeric value will be returned, like 0xFF0000. Default is rgb.
		 * @return {(array|number)} An array containing red, green, and blue (and optionally alpha) in that order, or if the format parameter was "hsl", the array will contain hue, saturation and lightness (and optionally alpha) in that order. Or if "format" is defined as "number", it'll return a number like 0xFF0000. Always numbers unless there's a relative prefix found in an hsl() or hsla() string and "format" is "hsl".
		 */
		_parseColor = function(v, format) {
			var toHSL = (format === "hsl"),
				a, r, g, b, h, s, l, max, min, d, wasHSL;
			if (!v) {
				a = _colorLookup.black;
			} else if (typeof(v) === "number") {
				a = [v >> 16, (v >> 8) & 255, v & 255];
			} else {
				if (v.charAt(v.length - 1) === ",") { //sometimes a trailing comma is included and we should chop it off (typically from a comma-delimited list of values like a textShadow:"2px 2px 2px blue, 5px 5px 5px rgb(255,0,0)" - in this example "blue," has a trailing comma. We could strip it out inside parseComplex() but we'd need to do it to the beginning and ending values plus it wouldn't provide protection from other potential scenarios like if the user passes in a similar value.
					v = v.substr(0, v.length - 1);
				}
				if (_colorLookup[v]) {
					a = _colorLookup[v];
				} else if (v.charAt(0) === "#") {
					if (v.length === 4) { //for shorthand like #9F0
						r = v.charAt(1);
						g = v.charAt(2);
						b = v.charAt(3);
						v = "#" + r + r + g + g + b + b;
					}
					v = parseInt(v.substr(1), 16);
					a = [v >> 16, (v >> 8) & 255, v & 255];
				} else if (v.substr(0, 3) === "hsl") {
					a = wasHSL = v.match(_numExp);
					if (!toHSL) {
						h = (Number(a[0]) % 360) / 360;
						s = Number(a[1]) / 100;
						l = Number(a[2]) / 100;
						g = (l <= 0.5) ? l * (s + 1) : l + s - l * s;
						r = l * 2 - g;
						if (a.length > 3) {
							a[3] = Number(v[3]);
						}
						a[0] = _hue(h + 1 / 3, r, g);
						a[1] = _hue(h, r, g);
						a[2] = _hue(h - 1 / 3, r, g);
					} else if (v.indexOf("=") !== -1) { //if relative values are found, just return the raw strings with the relative prefixes in place.
						return v.match(_relNumExp);
					}
				} else {
					a = v.match(_numExp) || _colorLookup.transparent;
				}
				a[0] = Number(a[0]);
				a[1] = Number(a[1]);
				a[2] = Number(a[2]);
				if (a.length > 3) {
					a[3] = Number(a[3]);
				}
			}
			if (toHSL && !wasHSL) {
				r = a[0] / 255;
				g = a[1] / 255;
				b = a[2] / 255;
				max = Math.max(r, g, b);
				min = Math.min(r, g, b);
				l = (max + min) / 2;
				if (max === min) {
					h = s = 0;
				} else {
					d = max - min;
					s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
					h = (max === r) ? (g - b) / d + (g < b ? 6 : 0) : (max === g) ? (b - r) / d + 2 : (r - g) / d + 4;
					h *= 60;
				}
				a[0] = (h + 0.5) | 0;
				a[1] = (s * 100 + 0.5) | 0;
				a[2] = (l * 100 + 0.5) | 0;
			}
			return (format === "number") ? (a[0] << 16 | a[1] << 8 | a[2]) : a;
		},
		_formatColors = function(s, toHSL) {
			var colors = (s + "").match(_colorExp) || [],
				charIndex = 0,
				parsed = "",
				i, color, temp;
			if (!colors.length) {
				return s;
			}
			for (i = 0; i < colors.length; i++) {
				color = colors[i];
				temp = s.substr(charIndex, s.indexOf(color, charIndex)-charIndex);
				charIndex += temp.length + color.length;
				color = _parseColor(color, (toHSL ? "hsl" : "rgb"));
				if (color.length === 3) {
					color.push(1);
				}
				parsed += temp + (toHSL ? "hsla(" + color[0] + "," + color[1] + "%," + color[2] + "%," + color[3] : "rgba(" + color.join(",")) + ")";
			}
			return parsed + s.substr(charIndex);
		}, _colorStringFilter,
		TweenLite = (_gsScope.GreenSockGlobals || _gsScope).TweenLite,
		_colorExp = "(?:\\b(?:(?:rgb|rgba|hsl|hsla)\\(.+?\\))|\\B#(?:[0-9a-f]{3}){1,2}\\b", //we'll dynamically build this Regular Expression to conserve file size. After building it, it will be able to find rgb(), rgba(), # (hexadecimal), and named color values like red, blue, purple, etc.

		_idMatrix = [1,0,0,0,0,0,1,0,0,0,0,0,1,0,0,0,0,0,1,0],
		_lumR = 0.212671,
		_lumG = 0.715160,
		_lumB = 0.072169,

		_applyMatrix = function(m, m2) {
			var temp = [],
				i = 0,
				z = 0,
				y, x;
			for (y = 0; y < 4; y++) {
				for (x = 0; x < 5; x++) {
					z = (x === 4) ? m[i + 4] : 0;
					temp[i + x] = m[i]   * m2[x] + m[i+1] * m2[x + 5] +	m[i+2] * m2[x + 10] + m[i+3] * m2[x + 15] +	z;
				}
				i += 5;
			}
			return temp;
		},

		_setSaturation = function(m, n) {
			var inv = 1 - n,
				r = inv * _lumR,
				g = inv * _lumG,
				b = inv * _lumB;
			return _applyMatrix([r + n, g, b, 0, 0, r, g + n, b, 0, 0, r, g, b + n, 0, 0, 0, 0, 0, 1, 0], m);
		},

		_colorize = function(m, color, amount) {
			var c = _parseColor(color),
				r = c[0] / 255,
				g = c[1] / 255,
				b = c[2] / 255,
				inv = 1 - amount;
			return _applyMatrix([inv + amount * r * _lumR, amount * r * _lumG, amount * r * _lumB, 0, 0, amount * g * _lumR, inv + amount * g * _lumG, amount * g * _lumB, 0, 0, amount * b * _lumR, amount * b * _lumG, inv + amount * b * _lumB, 0, 0, 0, 0, 0, 1, 0], m);
		},

		_setHue = function(m, n) {
			n *= Math.PI / 180;
			var c = Math.cos(n),
				s = Math.sin(n);
			return _applyMatrix([(_lumR + (c * (1 - _lumR))) + (s * (-_lumR)), (_lumG + (c * (-_lumG))) + (s * (-_lumG)), (_lumB + (c * (-_lumB))) + (s * (1 - _lumB)), 0, 0, (_lumR + (c * (-_lumR))) + (s * 0.143), (_lumG + (c * (1 - _lumG))) + (s * 0.14), (_lumB + (c * (-_lumB))) + (s * -0.283), 0, 0, (_lumR + (c * (-_lumR))) + (s * (-(1 - _lumR))), (_lumG + (c * (-_lumG))) + (s * _lumG), (_lumB + (c * (1 - _lumB))) + (s * _lumB), 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1], m);
		},

		_setContrast = function(m, n) {
			return _applyMatrix([n,0,0,0,0.5 * (1 - n), 0,n,0,0,0.5 * (1 - n), 0,0,n,0,0.5 * (1 - n), 0,0,0,1,0], m);
		},

		_getFilter = function(t, type) {
			var filterClass = _gsScope.PIXI.filters[type],
				filters = t.filters || [],
				i = filters.length,
				filter;
			if (!filterClass) {
				throw("PixiPlugin error: " + type + " isn't present.");
			}
			while (--i > -1) {
				if (filters[i] instanceof filterClass) {
					return filters[i];
				}
			}
			filter = new filterClass();
			if (type === "BlurFilter") {
				filter.blur = 0;
			}
			filters.push(filter);
			t.filters = filters;
			return filter;
		},

		_addColorMatrixFilterCacheTween = function(p, pg, cache, vars) { //we cache the ColorMatrixFilter components in a _gsColorMatrixFilter object attached to the target object so that it's easy to grab the current value at any time.
			pg._addTween(cache, p, cache[p], vars[p], p);
			pg._overwriteProps.push(p);
		},

		_applyBrightnessToMatrix = function(brightness, matrix) {
			var temp = new _gsScope.PIXI.filters.ColorMatrixFilter();
			temp.matrix = matrix;
			temp.brightness(brightness, true);
			return temp.matrix;
		},

		_CMFdefaults = {contrast:1, saturation:1, colorizeAmount:0, colorize:"rgb(255,255,255)", hue:0, brightness:1},

		_parseColorMatrixFilter = function(t, v, pg) {
			var filter = _getFilter(t, "ColorMatrixFilter"),
				cache = t._gsColorMatrixFilter = t._gsColorMatrixFilter || {contrast:1, saturation:1, colorizeAmount:0, colorize:"rgb(255,255,255)", hue:0, brightness:1},
				combine = v.combineCMF && !("colorMatrixFilter" in v && !v.colorMatrixFilter),
				i, matrix, startMatrix;
			startMatrix = filter.matrix;
			if (v.resolution) {
				filter.resolution = v.resolution;
			}
			if (v.matrix && v.matrix.length === startMatrix.length) {
				matrix = v.matrix;
				if (cache.contrast !== 1) {
					_addColorMatrixFilterCacheTween("contrast", pg, cache, _CMFdefaults);
				}
				if (cache.hue) {
					_addColorMatrixFilterCacheTween("hue", pg, cache, _CMFdefaults);
				}
				if (cache.brightness !== 1) {
					_addColorMatrixFilterCacheTween("brightness", pg, cache, _CMFdefaults);
				}
				if (cache.colorizeAmount) {
					_addColorMatrixFilterCacheTween("colorize", pg, cache, _CMFdefaults);
					_addColorMatrixFilterCacheTween("colorizeAmount", pg, cache, _CMFdefaults);
				}
				if (cache.saturation !== 1) {
					_addColorMatrixFilterCacheTween("saturation", pg, cache, _CMFdefaults);
				}

			} else {
				matrix = _idMatrix.slice();
				if (v.contrast != null) {
					matrix = _setContrast(matrix, Number(v.contrast));
					_addColorMatrixFilterCacheTween("contrast", pg, cache, v);
				} else if (cache.contrast !== 1) {
					if (combine) {
						matrix = _setContrast(matrix, cache.contrast);
					} else {
						_addColorMatrixFilterCacheTween("contrast", pg, cache, _CMFdefaults);
					}
				}
				if (v.hue != null) {
					matrix = _setHue(matrix, Number(v.hue));
					_addColorMatrixFilterCacheTween("hue", pg, cache, v);
				} else if (cache.hue) {
					if (combine) {
						matrix = _setHue(matrix, cache.hue);
					} else {
						_addColorMatrixFilterCacheTween("hue", pg, cache, _CMFdefaults);
					}
				}
				if (v.brightness != null) {
					matrix = _applyBrightnessToMatrix(Number(v.brightness), matrix);
					_addColorMatrixFilterCacheTween("brightness", pg, cache, v);
				} else if (cache.brightness !== 1) {
					if (combine) {
						matrix = _applyBrightnessToMatrix(cache.brightness, matrix);
					} else {
						_addColorMatrixFilterCacheTween("brightness", pg, cache, _CMFdefaults);
					}
				}
				if (v.colorize != null) {
					v.colorizeAmount = ("colorizeAmount" in v) ? Number(v.colorizeAmount) : 1;
					matrix = _colorize(matrix, v.colorize, v.colorizeAmount);
					_addColorMatrixFilterCacheTween("colorize", pg, cache, v);
					_addColorMatrixFilterCacheTween("colorizeAmount", pg, cache, v);
				} else if (cache.colorizeAmount) {
					if (combine) {
						matrix = _colorize(matrix, cache.colorize, cache.colorizeAmount);
					} else {
						_addColorMatrixFilterCacheTween("colorize", pg, cache, _CMFdefaults);
						_addColorMatrixFilterCacheTween("colorizeAmount", pg, cache, _CMFdefaults);
					}
				}
				if (v.saturation != null) {
					matrix = _setSaturation(matrix, Number(v.saturation));
					_addColorMatrixFilterCacheTween("saturation", pg, cache, v);
				} else if (cache.saturation !== 1) {
					if (combine) {
						matrix = _setSaturation(matrix, cache.saturation);
					} else {
						_addColorMatrixFilterCacheTween("saturation", pg, cache, _CMFdefaults);
					}
				}
			}
			i = matrix.length;
			while (--i > -1) {
				if (matrix[i] !== startMatrix[i]) {
					pg._addTween(startMatrix, i, startMatrix[i], matrix[i], "colorMatrixFilter");
				}
			}
			pg._overwriteProps.push("colorMatrixFilter");
		},

		_addColorTween = function(target, p, value, colorSetter, plugin) {
			var pt = colorSetter._firstPT = {_next:colorSetter._firstPT, t:target, p:p, proxy:{}, f:(typeof(target[p]) === "function")};
			pt.proxy[p] = "rgb(" + _parseColor(!pt.f ? target[p] : target[ ((p.indexOf("set") || typeof(target["get" + p.substr(3)]) !== "function") ? p : "get" + p.substr(3)) ]()).join(",") + ")";
			plugin._addTween(pt.proxy, p, "get", ((typeof(value) === "number") ? "rgb(" + _parseColor(value, false).join(",") + ")" : value), p, null, null, _colorStringFilter);
		},

		//to improve performance, when a color is sensed, we hijack the setRatio() method of the plugin instance with a new function that this method spits back. This is a special method that handles parsing color values on-the-fly and turns them into numeric values which PixiJS requires. In other words, instead of "rgb(255, 0, 0)", PixiJS wants 0xFF0000. This also works with hsl() values.
		_buildColorSetter = function(tween, plugin) {
			var setRatio = plugin.setRatio, //save the original (super) setRatio() function
				func = function(v) {
					var pt = func._firstPT,
						val;
					setRatio.call(plugin, v);
					while (pt) {
						val = _parseColor(pt.proxy[pt.p], "number");
						if (pt.f) {
							pt.t[pt.p](val);
						} else {
							pt.t[pt.p] = val;
						}
						pt = pt._next;
					}
					if (func.graphics) { //in order for PixiJS to actually redraw GraphicsData, we've gotta increment the "dirty" and "clearDirty" values. If we don't do this, the values will be tween properly, but not rendered.
						func.graphics.dirty++;
						func.graphics.clearDirty++;
					}
				};
			plugin.setRatio = func;
			return func;
		},


		_colorProps = {tint:1, lineColor:1, fillColor:1},
		_xyContexts = "position,scale,skew,pivot,anchor,tilePosition,tileScale".split(","),
		_contexts = {x:"position", y:"position", tileX:"tilePosition", tileY:"tilePosition"},
		_colorMatrixFilterProps = {colorMatrixFilter:1, saturation:1, contrast:1, hue:1, colorize:1, colorizeAmount:1, brightness:1, combineCMF:1},
		_DEG2RAD = Math.PI / 180,
        _degreesToRadians = function(value) {
			return (typeof(value) === "string" && value.charAt(1) === "=") ? value.substr(0, 2) + (parseFloat(value.substr(2)) * _DEG2RAD) : value * _DEG2RAD;
        }, i, p;

	//context setup...
	for (i = 0; i < _xyContexts.length; i++) {
		p = _xyContexts[i];
		_contexts[p + "X"] = p;
		_contexts[p + "Y"] = p;
    }

    //color parsing setup...
	for (p in _colorLookup) {
		_colorExp += "|" + p + "\\b";
	}
	_colorExp = new RegExp(_colorExp+")", "gi");
	_colorStringFilter = function(a) {
		var combined = a[0] + " " + a[1],
			toHSL;
		_colorExp.lastIndex = 0;
		if (_colorExp.test(combined)) {
			toHSL = (combined.indexOf("hsl(") !== -1 || combined.indexOf("hsla(") !== -1);
			a[0] = _formatColors(a[0], toHSL);
			a[1] = _formatColors(a[1], toHSL);
		}
	};

	if (!TweenLite.defaultStringFilter) {
		TweenLite.defaultStringFilter = _colorStringFilter;
	}

    var PixiPlugin = _gsScope._gsDefine.plugin({
        propName: "pixi",
        priority: 0,
        API: 2,
		global: true,
        version: "0.2.1",

        init: function (target, values, tween, index) {
            if (!target instanceof _gsScope.PIXI.DisplayObject) {
                return false;
            }
            var context, axis, value, colorMatrix, filter, p, padding, colorSetter, i, data, pt;
            for (p in values) {
                context = _contexts[p];
                value = values[p];
                if (typeof(value) === "function") {
                    value = value(index || 0, target);
                }
                if (context) {
                    axis = (p.charAt(p.length-1).toLowerCase().indexOf("x") !== -1) ? "x" : "y";
					this._addTween(target[context], axis, target[context][axis], (context === "skew") ? _degreesToRadians(value) : value, p);
                } else if (p === "scale" || p === "anchor" || p === "pivot" || p === "tileScale") {
					this._addTween(target[p], "x", target[p].x, value, p + "X");
					this._addTween(target[p], "y", target[p].y, value, p + "Y");
                } else if (p === "rotation") { //PIXI expects rotation in radians, but as a convenience we let folks define it in degrees and we do the conversion.
					this._addTween(target, p, target.rotation, _degreesToRadians(value), p);

                } else if (_colorMatrixFilterProps[p]) {
					if (!colorMatrix) {
						_parseColorMatrixFilter(target, values.colorMatrixFilter || values, this);
						colorMatrix = true;
					}
                } else if (p === "blur" || p === "blurX" || p === "blurY" || p === "blurPadding") {
					filter = _getFilter(target, "BlurFilter");
					this._addTween(filter, p, filter[p], value, p);
					if (values.blurPadding !== 0) {
						padding = values.blurPadding || Math.max(filter[p], value) * 2;
						i = target.filters.length;
						while (--i > -1) {
							target.filters[i].padding = Math.max(target.filters[i].padding, padding); //if we don't expand the padding on all the filters, it can look clipped.
						}
					}
                } else if (_colorProps[p]) {
					if (!colorSetter) {
						colorSetter = _buildColorSetter(tween, this);
					}
					if ((p === "lineColor" || p === "fillColor") && target instanceof _gsScope.PIXI.Graphics) {
						data = target.graphicsData;
						i = data.length;
						while (--i > -1) {
							_addColorTween(data[i], p, value, colorSetter, this);
						}
						colorSetter.graphics = target;
					} else {
						_addColorTween(target, p, value, colorSetter, this);
					}
                } else if (p === "autoAlpha") {
					this._firstPT = pt = {t: {setRatio:function() { target.visible = !!target.alpha; }}, p: "setRatio", s: 0, c: 1, f: 1, pg: 0, n: "visible", pr: 0, m: 0, _next:this._firstPT};
					if (pt._next) {
						pt._next._prev = pt;
					}
					this._addTween(target, "alpha", target.alpha, value, "alpha");
					this._overwriteProps.push("alpha", "visible");
                } else {
					this._addTween(target, p, target[p], value, p);
                }
				this._overwriteProps.push(p);
            }
            return true;
        }
    });

	PixiPlugin.colorProps = _colorProps;
	PixiPlugin.parseColor = _parseColor;
	PixiPlugin.formatColors = _formatColors;
	PixiPlugin.colorStringFilter = _colorStringFilter;


}); if (_gsScope._gsDefine) { _gsScope._gsQueue.pop()(); }

//export to AMD/RequireJS and CommonJS/Node (precursor to full modular build system coming at a later date)
(function(name) {
	"use strict";
	var getGlobal = function() {
		return (_gsScope.GreenSockGlobals || _gsScope)[name];
	};
	if (typeof(module) !== "undefined" && module.exports) { //node
		require("gsap/TweenLite");
		module.exports = getGlobal();
	} else if (typeof(define) === "function" && define.amd) { //AMD
		define(["gsap/TweenLite"], getGlobal);
	}
}("PixiPlugin"));